//
//  SocketStreamClient.m
//  FullControl
//
//  Created by Francesco Burelli on 12/12/09.
//  Copyright 2009 Francesco Burelli. All rights reserved.
//

#import "SocketStreamClient.h"


@implementation SocketStreamClient
@synthesize delegate=_delegate, service=_service, iStream=_iStream, oStream=_oStream, connected=_connected;

-(id)init {
	self = [super init];
	_connected = NO;
	dataToSend = [[NSMutableData alloc] init];
	return self;
}
-(void)dealloc {
	[self disconnect];
	[dataToSend release];
	[super dealloc];
}

-(void)connectWithService:(NSNetService *)netService {
	// inform the delegate
	if (self.delegate && [self.delegate respondsToSelector:@selector(willConnectWithService:)]) { 
        [self.delegate willConnectWithService:netService];
    }
	
	// destroy all
	[self disconnect];
	
	// create the socket
	socket = CFSocketCreate(NULL, PF_INET, SOCK_STREAM, IPPROTO_TCP, kCFSocketNoCallBack, NULL, NULL);
	
	if (socket) {
		_iStream = NULL;
		_oStream = NULL;
		
		// connect the socket and get streams
		if([_service getInputStream:&_iStream outputStream:&_oStream]) {
			// save the service
			_service = netService;
			
			[_iStream retain];
			[_oStream retain];
			
			[_oStream setDelegate:self];
			[_iStream setDelegate:self];
			[_oStream scheduleInRunLoop:[NSRunLoop currentRunLoop] forMode:NSDefaultRunLoopMode];
			[_iStream scheduleInRunLoop:[NSRunLoop currentRunLoop] forMode:NSDefaultRunLoopMode];
			[_oStream open];
			[_iStream open];
			
		}
		else {
			// tell the error to delegate
			NSError *error;
			error = [NSError errorWithDomain:@"bonjour" code:1 userInfo:nil];
			if (self.delegate && [self.delegate respondsToSelector:@selector(didReceiveError:)]) { 
				[self.delegate didReceiveError:error];
			}
			// destroy socket and streams
			[self disconnect];
		}
	}
	else {
		// can't create socket, tell delegate
		NSError *error;
		error = [NSError errorWithDomain:@"socket" code:1 userInfo:nil];
		if (self.delegate && [self.delegate respondsToSelector:@selector(didReceiveError:)]) { 
			[self.delegate didReceiveError:error];
		}
	}
	
}

// -------------  STREAM DELEGATE ------------
- (void)stream:(NSStream *)stream handleEvent:(NSStreamEvent)eventCode {
	
	switch (eventCode) {
        case NSStreamEventOpenCompleted: {
			if(stream == _iStream)
				input = YES;
			
			if(stream == _oStream)
				output = YES;
			
			if(input && output) {
				_connected = YES;
				// inform delegate that connection success
				if (self.delegate && [self.delegate respondsToSelector:@selector(didConnectWithService:)]) { 
					[self.delegate didConnectWithService:self.service];
				}
			}
			
        } break;
        case NSStreamEventHasBytesAvailable: {
			NSMutableData *dataReceived = [[NSMutableData alloc] init];
			
			uint8_t buf[1024];
			unsigned int len = 0;
			len = [(NSInputStream *)stream read:buf maxLength:1024];
			if ( len ) {
				[dataReceived appendBytes:(const void *)buf length:len];
				// give received data to delegate
				if (self.delegate && [self.delegate respondsToSelector:@selector(didReceiveData:from:)]) { 
					[self.delegate didReceiveData:(NSData *)dataReceived from:self.service];
				}
				
			}
			
			[dataReceived release];
			dataReceived = nil;
			
			break;
			
        }
        case NSStreamEventHasSpaceAvailable: {
			// if any, send data remaining
			if (dataToSend && [dataToSend length]) {
				NSInteger bytesWrite;
				bytesWrite = [_oStream write:[dataToSend bytes] maxLength:[dataToSend length]];
				
				if (bytesWrite == [dataToSend length]) {
					[dataToSend initWithLength:0];
				}
				else {
					NSRange range = { bytesWrite, [dataToSend length] - bytesWrite};
					[dataToSend initWithData:[dataToSend subdataWithRange:range]];
				}
				
			}
			
			break;
			
        }
        case NSStreamEventErrorOccurred: {
			NSError *streamError;
			streamError = [stream streamError];
			if (self.delegate && [self.delegate respondsToSelector:@selector(didReceiveError:)]) { 
				[self.delegate didReceiveError:streamError];
			}
			
			
		} break;
        case NSStreamEventEndEncountered: {
			[self disconnect];
			
        } break;
	}
	
}

-(void)disconnect {
	if (_iStream) {
		[_iStream close];
		[_iStream removeFromRunLoop:[NSRunLoop currentRunLoop] forMode:NSDefaultRunLoopMode];
		[_iStream release];
		_iStream = nil;
	}
	if (_oStream) {
		[_oStream close];
		[_oStream removeFromRunLoop:[NSRunLoop currentRunLoop] forMode:NSDefaultRunLoopMode];
		[_oStream release];
		_oStream = nil;
	}
	if (socket) {
		CFSocketInvalidate(socket);
		CFRelease(socket);
		socket = NULL;
	}
	if (self.delegate && [self.delegate respondsToSelector:@selector(didNotConnectWithService:)]) { 
		[self.delegate didNotConnectWithService:_service];
	}
	
}
-(NSInteger)sendData:(NSData *)theData {
	NSInteger bytesWrite;
	
	if ([_oStream hasSpaceAvailable]) {
		bytesWrite = [_oStream write:[theData bytes] maxLength:[theData length]];
		if (bytesWrite != [theData length]) {
			// if can't send all, store part of data and send when _oStream hasSpaceAvailable
			NSRange range = { bytesWrite, [theData length] - bytesWrite};
			[dataToSend appendData:[theData subdataWithRange:range]];
		}
	}
	else {
		// if can't send, store data and send when _oStream hasSpaceAvailable
		[dataToSend appendData:theData];
	}
	return bytesWrite;
}

@end
